<?php
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php_errors.log');

date_default_timezone_set('Asia/Tehran');

define('SECURE_ACCESS', true);

$config = require '/home1/footbal1/secure_config/config.php';
require_once 'functions.php';

try {
    $db = new PDO(
        "mysql:host={$config['db_host']};dbname={$config['db_name']};charset=utf8mb4",
        $config['db_user'],
        $config['db_pass']
    );
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die("⛔ اتصال به پایگاه داده ناموفق بود.");
}

// دریافت بازی‌های آماده پردازش
$stmt = $db->prepare("
    SELECT id, result, team_home, team_away
    FROM matches
    WHERE result IS NOT NULL AND pool_processed = 0
");
$stmt->execute();
$matches = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (!$matches) {
    exit("✅ هیچ بازی آماده پردازش نیست.\n");
}

foreach ($matches as $match) {
    $match_id = $match['id'];
    $result = $match['result'];
    $home = getTeamName($match['team_home']);
    $away = getTeamName($match['team_away']);

    $stmt = $db->prepare("SELECT id, user_id, tokens, option_selected FROM pool_investments WHERE match_id = ?");
    $stmt->execute([$match_id]);
    $investments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (!$investments) {
        markMatchAsProcessed($match_id);
        continue;
    }

    $total_pool = 0;
    $total_winner_invest = 0;
    $options_invest = [];  // مجموع سرمایه هر گزینه

    foreach ($investments as $inv) {
        $option = $inv['option_selected'];
        $options_invest[$option] = ($options_invest[$option] ?? 0) + $inv['tokens'];
        $total_pool += $inv['tokens'];
        if ($option === $result) {
            $total_winner_invest += $inv['tokens'];
        }
    }

    // 🔥 بررسی تمرکز یک‌طرفه
    $is_single_option = (count($options_invest) === 1);
    $single_option_key = array_key_first($options_invest);
    
    if ($is_single_option && $result === $single_option_key) {
    
        $guaranteed_percent = 0.15;  // مقدار سود تضمینی هر نفر (۱۵٪)
        $db->beginTransaction();
    
        try {
            $total_guaranteed_paid = 0;  // مجموع سود پرداخت‌شده از صندوق
    
            foreach ($investments as $inv) {
                $user_id = $inv['user_id'];
                $tokens = $inv['tokens'];
    
                $guaranteed_profit = round($tokens * $guaranteed_percent, 2);
                $total_payout = $tokens + $guaranteed_profit;
    
                // واریز به حساب کاربر
                $stmt = $db->prepare("UPDATE users SET score = score + ? WHERE user_id = ?");
                $stmt->execute([$total_payout, $user_id]);
    
                // ثبت در جدول pool_investments
                $stmt = $db->prepare("UPDATE pool_investments SET is_closed = 1, is_winner = 1, tokens_won = ? WHERE id = ?");
                $stmt->execute([$total_payout, $inv['id']]);
                
                $message = "🎉 تبریک! در حالت یک‌طرفه بردی.\n"
                         . "🏟️ $home vs $away\n"
                         . "🔹 سود تضمینی: {$guaranteed_profit} توکن\n"
                         . "💰 مجموع پرداختی: {$total_payout} توکن";
                queueNotification($user_id, $message);
    
            // دریافت موجودی صندوق سیستم
            $stmt = $db->prepare("SELECT amount FROM system_fund WHERE id = 1 LIMIT 1");
            $stmt->execute();
            $system_fund = (int)$stmt->fetchColumn();

                // بروزرسانی موجودی صندوق سیستم
            $system_fund -= $guaranteed_profit;
            $total_guaranteed_paid += $guaranteed_profit;
            }
    
            // ثبت موجودی جدید صندوق
            $stmt = $db->prepare("UPDATE system_fund SET amount = ? WHERE id = 1");
            $stmt->execute([$system_fund]);
    
            // ثبت برداشت از صندوق در جدول pool_commissions
            $stmt = $db->prepare("INSERT INTO pool_commissions (match_id, commission_amount, type, created_at) VALUES (?, ?, 'system_fund_use', NOW())");
            $stmt->execute([$match_id, $total_guaranteed_paid]);
    
            $db->commit();
            markMatchAsProcessed($match_id);
            continue;  // بازی تمام شد، برو سراغ بازی بعدی
    
        } catch (Exception $e) {
            $db->rollBack();
            exit("❌ خطا در پرداخت سود تضمینی استخر بازی $match_id: " . $e->getMessage());
        }
    }


    // 🔥 حالت عادی (غیر یک‌طرفه)
    $commission = $total_pool * 0.10;
    $stmt = $db->prepare("INSERT INTO pool_commissions (match_id, commission_amount, type, created_at) VALUES (?, ?, 'normal', NOW())");
    $stmt->execute([$match_id, $commission]);

    $final_pool = $total_pool - $commission;

    $db->beginTransaction();

    try {

        if ($total_winner_invest > 0) {
            foreach ($investments as $inv) {
                $user_id = $inv['user_id'];
                $user_invest = $inv['tokens'];

                if ($inv['option_selected'] === $result) {
                    $user_share = round(($user_invest / $total_winner_invest) * $final_pool, 2);

                    $stmt = $db->prepare("UPDATE users SET score = score + ? WHERE user_id = ?");
                    $stmt->execute([$user_share, $user_id]);

                    $stmt = $db->prepare("UPDATE pool_investments SET is_closed = 1, is_winner = 1, tokens_won = ? WHERE id = ?");
                    $stmt->execute([$user_share, $inv['id']]);
                    $message = "🎉 تبریک! سرمایه‌گذاری‌ات نتیجه داد.\n"
                             . "🏟️ $home vs $away\n"
                             . "🔹 نتیجه واقعی: " . matchText($result, $home, $away) . "\n"
                             . "💰 مبلغ برد: {$user_share} توکن\n\n"
                             . "👏 آفرین! برو سراغ بازی‌های بعدی.";
                    queueNotification($user_id, $message);


                } else {
                    $stmt = $db->prepare("UPDATE pool_investments SET is_closed = 1, is_winner = 0 WHERE id = ?");
                    $stmt->execute([$inv['id']]);

                    $message = "❌ سرمایه‌گذاریت به پایان رسید و متاسفانه بازنده شدی.\n"
                             . "🏟️ $home vs $away\n"
                             . "🔹 نتیجه واقعی: " . matchText($result, $home, $away) . "\n\n"
                             . "😢 شانس باهات یار نبود. دفعه بعد جبران کن.";
                    queueNotification($user_id, $message);
                }
            }

        } else {
            // 🔥 حالت بدون برنده => سود سیستم
            foreach ($investments as $inv) {
                $stmt = $db->prepare("UPDATE pool_investments SET is_closed = 1, is_winner = 0 WHERE id = ?");
                $stmt->execute([$inv['id']]);
                queueNotification($inv['user_id'], "❌ نتیجه سرمایه‌گذاری شما درست از آب درنیومد. امیدوارم دفعه بعد موفق‌تر باشید.");
            }

            $stmt = $db->prepare("SELECT amount FROM system_fund WHERE id = 1 LIMIT 1");
            $stmt->execute();
            $system_fund = (int)$stmt->fetchColumn();

            $system_fund += $total_pool;
            $stmt = $db->prepare("UPDATE system_fund SET amount = ? WHERE id = 1");
            $stmt->execute([$system_fund]);

            $stmt = $db->prepare("INSERT INTO pool_commissions (match_id, commission_amount, type, created_at) VALUES (?, ?, 'system_profit', NOW())");
            $stmt->execute([$match_id, $total_pool]);
        }

        $db->commit();

    } catch (Exception $e) {
        $db->rollBack();
        exit("❌ خطا در پردازش استخر بازی $match_id: " . $e->getMessage());
    }

    markMatchAsProcessed($match_id);
}

exit("🎉 همه استخرهای آماده پردازش شدند.\n");

// ======= توابع کمکی =======

function markMatchAsProcessed($match_id) {
    global $db;
    $stmt = $db->prepare("UPDATE matches SET pool_processed = 1 WHERE id = ?");
    $stmt->execute([$match_id]);
}

function queueNotification($user_id, $message) {
    global $db;

    // 🔒 گارد: اگر user_id نامعتبر بود، اصلاً چیزی وارد DB نشه
    if (!is_numeric($user_id) || (int)$user_id <= 0) {
        error_log('[queueNotification] invalid user_id = ' . print_r($user_id, true));
        return;
    }

    $stmt = $db->prepare("INSERT INTO investment_notifications (user_id, message) VALUES (?, ?)");
    $stmt->execute([(int)$user_id, $message]);
}
