<?php

if (!defined('APP_BOOTSTRAPPED')) {
    require __DIR__ . '/../_guard.php';
}

date_default_timezone_set("Asia/Tehran");

function showActivePredictions($chat_id) {
    global $db;

    $stmt = $db->prepare("SELECT m.team_home, m.team_away, m.match_time, p.prediction 
                      FROM predictions p
                      JOIN matches m ON p.match_id = m.id
                      WHERE p.user_id = ? AND m.result IS NULL
                      ORDER BY m.match_time ASC");
    $stmt->execute([$chat_id]);
    $rows = $stmt->fetchAll();

    if (count($rows) == 0) {
        sendMessage($chat_id, "⛔️ هیچ پیش‌بینی فعالی نداری.");
        return;
    }

    $msg = "📅 <b>مشاهده لیست پیش‌بینی‌های امروزت</b>\n\n";
    foreach ($rows as $row) {
        $team_home = getTeamName($row['team_home']);
        $team_away = getTeamName($row['team_away']);
        $msg .= "▫️ {$team_home} vs {$team_away}\n";
        $msg .= "🔮 پیش‌بینی شما: " . matchText($row['prediction'], $team_home, $team_away) . "\n\n";
    }

    // دکمه بازگشت
    $keyboard = [
        'inline_keyboard' => [
            [['text' => '🔙 بازگشت', 'callback_data' => 'profile_menu']]
        ]
    ];

    file_get_contents("https://api.telegram.org/bot{$GLOBALS['config']['token']}/sendMessage?" . http_build_query([
        'chat_id' => $chat_id,
        'text' => $msg,
        'reply_markup' => json_encode($keyboard),
        'parse_mode' => 'HTML'
    ]));
}

function showPastPredictions($chat_id) {
    global $db;

    $stmt = $db->prepare("
        SELECT p.prediction, m.team_home, m.team_away, m.match_time, m.result
        FROM predictions p
        JOIN matches m ON p.match_id = m.id
        WHERE p.user_id = ?
          AND m.result IS NOT NULL
          AND DATE(m.match_time) = CURDATE()
        ORDER BY m.match_time DESC
        LIMIT 20
    ");

    $stmt->execute([$chat_id]);
    $rows = $stmt->fetchAll();

    if (empty($rows)) {
        sendMessage($chat_id, "📭 هنوز هیچ بازی‌ای نتیجه‌اش مشخص نشده.");
        return;
    }

    $msg = "📈 نتایج پیش‌بینی‌های شما:\n\n";
    foreach ($rows as $r) {
        $team_home = getTeamName($r['team_home']);
        $team_away = getTeamName($r['team_away']);
        
        $your_prediction = matchText($r['prediction'], $team_home, $team_away);
        $real_result = matchText($r['result'], $team_home, $team_away);
        $status = ($r['prediction'] === $r['result']) ? "✅ درست" : "❌ اشتباه";

        $line = "📜 {$team_home} vs {$team_away}\n";
        $line .= "📅 " . date("Y-m-d", strtotime($r['match_time'])) . "\n";
        $line .= "🔸 پیش‌بینی شما: $your_prediction\n";
        $line .= "🔹 نتیجه واقعی: $real_result\n";
        $line .= "🎯 وضعیت: $status\n\n";

        if (mb_strlen($msg . $line) >= 3500) {
            sendMessage($chat_id, $msg);
            $msg = "📈 نتایج پیش‌بینی‌های شما:\n\n";
        }

        $msg .= $line;
    }

    if (mb_strlen(trim($msg)) > 0) {
        $keyboard = [
            'inline_keyboard' => [
                [['text' => '🔙 بازگشت', 'callback_data' => 'start']]
            ]
        ];

        file_get_contents("https://api.telegram.org/bot{$GLOBALS['config']['token']}/sendMessage?" . http_build_query([
            'chat_id' => $chat_id,
            'text' => $msg,
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'HTML'
        ]));
    }
}

function showLeaguesForPrediction($chat_id) {
    global $db;
    
    $stmt = $db->query("SELECT id, name FROM leagues ORDER BY id ASC");
    $leagues = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($leagues)) {
        sendMessage($chat_id, "📅 در حال حاضر لیگی برای پیش‌بینی ثبت نشده.");
    } else {
        $buttons = [];
        foreach ($leagues as $league) {
            $buttons[] = [["text" => $league['name'], "callback_data" => "league_" . $league['id']]];
        }
        sendInlineKeyboard($chat_id, "🏆 لطفاً لیگ مورد نظر خود را انتخاب کنید:", $buttons);
    }
}

function showLeagueMatches($chat_id, $league_id) {
    global $db;
    
    $stmt = $db->prepare("SELECT id, team_home, team_away, match_time FROM matches WHERE league_id = ? AND match_day = CURDATE()");
    $stmt->execute([$league_id]);
    $matches = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($matches)) {
        sendMessage($chat_id, "📅 بازیی برای امروز وجود ندارد.");
    } else {
        $buttons = [];
        foreach ($matches as $match) {
            $text = $match['team_home'] . " vs " . $match['team_away'];
            $time = date("H:i", strtotime($match['match_time']));
            $callback = "match_" . $match['id'];
            $buttons[] = [["text" => "$text ($time)", "callback_data" => $callback]];
        }
        sendInlineKeyboard($chat_id, "🎯 یکی از بازی‌های امروز رو برای پیش‌بینی انتخاب کن:", $buttons);
    }
}

function showMatchPredictionOptions($chat_id, $match_id) {
    global $db;
    
    $stmt = $db->prepare("SELECT team_home, team_away FROM matches WHERE id = ?");
    $stmt->execute([$match_id]);
    $match = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($match) {
        $home = $match['team_home'];
        $away = $match['team_away'];
        $buttons = [
            [["text" => "🏠 برد $home", "callback_data" => "predict_{$match_id}_home"]],
            [["text" => "🤝 مساوی", "callback_data" => "predict_{$match_id}_draw"]],
            [["text" => "🛫 برد $away", "callback_data" => "predict_{$match_id}_away"]]
        ];
        sendInlineKeyboard($chat_id, "📢 نتیجه مورد نظر خودت رو انتخاب کن:", $buttons);
    }
}

function savePrediction($chat_id, $match_id, $prediction) {
    global $db;

    // بررسی زمان بازی
    $stmt = $db->prepare("SELECT match_time FROM matches WHERE id = ?");
    $stmt->execute([$match_id]);
    $match = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$match) {
        sendMessage($chat_id, "❌ بازی‌ای با این آی‌دی پیدا نشد.");
        return;
    }

    $match_time = new DateTime($match['match_time'], new DateTimeZone('UTC'));
    $now = new DateTime("now", new DateTimeZone('UTC'));

    // بررسی پیش‌بینی تکراری
    $check = $db->prepare("SELECT id FROM predictions WHERE user_id = ? AND match_id = ?");
    $check->execute([$chat_id, $match_id]);

    if ($check->rowCount() > 0) {
        sendMessage($chat_id, "⚠️ شما قبلاً برای این بازی پیش‌بینی انجام داده‌اید.");
        return;
    }

    // بررسی انقضای زمان بازی
    if ($match_time <= $now) {
        sendMessage($chat_id, "⏰ مهلت پیش‌بینی این بازی تمام شده.");
        return;
    }

    // 🔐 بررسی فاصله زمانی از آخرین پیش‌بینی
    $stmt = $db->prepare("SELECT created_at FROM predictions WHERE user_id = ? ORDER BY created_at DESC LIMIT 1");
    $stmt->execute([$chat_id]);
    $last_prediction_time = $stmt->fetchColumn();

    if ($last_prediction_time) {
        $last_time = strtotime($last_prediction_time);
        $current_time = time();
        $diff = $current_time - $last_time;

        if ($diff < 5) {
            logMessage("SUSPICIOUS", "User $chat_id submitted prediction after $diff seconds.");
            // فقط لاگ → بدون محدودسازی
        }
    }

    // ثبت پیش‌بینی
    $insert = $db->prepare("INSERT INTO predictions (user_id, match_id, prediction) VALUES (?, ?, ?)");
    if ($insert->execute([$chat_id, $match_id, $prediction])) {
        sendMessage($chat_id, "✅ پیش‌بینی شما ثبت شد. موفق باشی!");

            // 📌 افزایش total_predictions
    $stmt = $db->prepare("UPDATE users SET total_predictions = total_predictions + 1 WHERE user_id = ?");
    $stmt->execute([$chat_id]);
    
    } else {
        sendMessage($chat_id, "❌ خطا در ثبت پیش‌بینی!");
    }
}

function showPredictionHistory($chat_id) {
    global $db;

    // دریافت ایمیل کاربر
    $stmt = $db->prepare("SELECT email FROM users WHERE user_id = ?");
    $stmt->execute([$chat_id]);
    $email = $stmt->fetchColumn();

    if (!$email) {
        sendMessage($chat_id, "📧 لطفاً ابتدا ایمیل خود را ثبت کن با دستور:\n/setemail your@email.com");
        return;
    }

    // گرفتن پیش‌بینی‌ها
    $stmt = $db->prepare("
        SELECT p.match_id, p.prediction, m.team_home, m.team_away, m.match_time, m.result
        FROM predictions p
        JOIN matches m ON p.match_id = m.id
        WHERE p.user_id = ? AND m.result IS NOT NULL
        ORDER BY m.match_time DESC
    ");
    $stmt->execute([$chat_id]);
    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($results)) {
        sendMessage($chat_id, "📭 هنوز هیچ بازی‌ای نتیجه‌اش مشخص نشده.");
        return;
    }

    // ساخت فایل CSV
    $file = __DIR__ . "/history_$chat_id.csv";
    $f = fopen($file, "w");
    fputcsv($f, ['Team Home', 'Team Away', 'Match Time', 'Your Prediction', 'Real Result', 'Status']);

    foreach ($results as $r) {
        $your_prediction = matchText($r['prediction'], $r['team_home'], $r['team_away']);
        $real_result = matchText($r['result'], $r['team_home'], $r['team_away']);
        $status = ($r['prediction'] === $r['result']) ? "✅ Correct" : "❌ Wrong";

        fputcsv($f, [$r['team_home'], $r['team_away'], $r['match_time'], $your_prediction, $real_result, $status]);
    }
    fclose($f);

    // ارسال ایمیل با فایل پیوست
    sendPredictionEmail($email, $file);
    unlink($file);
    sendMessage($chat_id, "📤 فایل تاریخچه به ایمیل شما ارسال شد: $email");
}

function sendPredictionEmail($email, $file_path) {
    $subject = "Your Prediction History";
    $message = "Hi,\n\nYour prediction history is attached.\n\n-- Football Bot";

    $separator = md5(time());
    $filename = basename($file_path);
    $filetype = mime_content_type($file_path);
    $attachment = chunk_split(base64_encode(file_get_contents($file_path)));

    // Header
    $headers = "From: Football Bot <noreply@yourdomain.com>\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: multipart/mixed; boundary=\"" . $separator . "\"\r\n";

    // Body
    $body = "--" . $separator . "\r\n";
    $body .= "Content-Type: text/plain; charset=\"UTF-8\"\r\n";
    $body .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
    $body .= $message . "\r\n";

    $body .= "--" . $separator . "\r\n";
    $body .= "Content-Type: " . $filetype . "; name=\"" . $filename . "\"\r\n";
    $body .= "Content-Transfer-Encoding: base64\r\n";
    $body .= "Content-Disposition: attachment; filename=\"" . $filename . "\"\r\n\r\n";
    $body .= $attachment . "\r\n";
    $body .= "--" . $separator . "--";

    // ارسال ایمیل
    mail($email, $subject, $body, $headers);
}
